<?php

namespace Mtc\MercuryDataModels;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Config;
use Mtc\ContentManager\Models\Comment;
use Mtc\MercuryDataModels\Contracts\ContentHistoryModel;
use Mtc\MercuryDataModels\Factories\DealershipContentHistoryFactory;

class DealershipContentHistory extends Model implements ContentHistoryModel
{
    use HasFactory;
    use SoftDeletes;

    protected $table = 'dealership_content_history';

    protected $fillable = [
        'uuid',
        'dealership_id',
        'author_id',
        'is_active',
        'pending_review',
        'request_changes',
        'data',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'data' => 'array',
    ];

    protected static function newFactory()
    {
        return DealershipContentHistoryFactory::new();
    }

    public function dealership(): BelongsTo
    {
        return $this->belongsTo(Dealership::class);
    }

    public function author(): BelongsTo
    {
        return $this->belongsTo(User::class, 'author_id');
    }

    /**
     * Relationship with content elements on version
     *
     * @return MorphMany
     */
    public function content(): MorphMany
    {
        return $this->morphMany(Config::get('pages.version_content_model'), 'owner')
            ->whereNull('parent_id')
            ->orderBy('order');
    }

    /**
     * Relationship with content elements on version
     *
     * @return MorphMany
     */
    public function allContent(): MorphMany
    {
        return $this->morphMany(Config::get('pages.version_content_model'), 'owner')
            ->orderBy('order');
    }

    public function scopeWithCommentCount(Builder $query): void
    {
        $query->addSelect([
            'comment_count' => Comment::query()
                ->selectRaw('COUNT(*)')
                ->where('commentable_type', $this->getMorphClass())
                ->whereColumn('commentable_id', $this->getTable() . '.id')
                ->whereNull('resolved_at')
        ]);
    }

    public function url(): string
    {
        $prefix = app()->environment('local') ? 'http://' : 'https://';
        return tenant()->primaryDomain
            ? $prefix . tenant()->primaryDomain->domain . $this->offer->urlPath() . '?v=' . $this->uuid
            : $prefix . tenant()->domains()->first()->domain . $this->offer->urlPath() . '?v=' . $this->uuid;
    }

    public function getStatusAttribute(): string
    {
        if ($this->attributes['request_changes']) {
            return __('pages.changes_requested');
        }

        if ($this->attributes['pending_review']) {
            return __('pages.pending_review');
        }

        return $this->attributes['is_active']
            ? __('pages.is_active')
            : __('pages.draft');
    }
}
