<?php

namespace Mtc\MercuryDataModels\Filters;

use App\Facades\Settings;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;

class PricePointFilter extends IsFilter
{
    // allow this filter to be shown when only one option is available.
    // we may not want to make all price indicator options visible (e.g. "poor"),
    // however we still include "poor" price vehicles in the results.
    protected int $min_count_to_show = 1;

    /**
     * Apply selections to current filtered object
     *
     * @param Builder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selection = [])
    {
        $query->whereHas('autoTraderData', function ($relation_query) use ($selection) {
            $relation_query->whereIn('price_point', $selection);
        });
    }

    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        $price_points_results = $this->getQuery()
            ->distinct()
            ->join('vehicle_autotrader_data', 'vehicles.id', 'vehicle_autotrader_data.vehicle_id')
            ->when(Settings::get('filter-apply-selections-to-results'), fn($query) => $query->where($product_filtering))
            ->whereIn('vehicle_autotrader_data.price_point', $this->getAllowedOptions())
            ->select('price_point')
            ->distinct()
            ->get()
            ->each(function ($value) {
                // force labels to upper case first letter of words
                $value->price_point = ucwords(strtolower($value->price_point));
            });

        $ordered_options_to_return = collect([]);

        // iterate over the allowed options and add the model with the matching name to the collection to be returned.
        // this ensures that option models are returned in the required order.
        collect($this->getAllowedOptions())
            ->each(function ($allowed_option) use ($ordered_options_to_return, $price_points_results) {
                $item_to_push = $price_points_results->first(function ($item) use ($allowed_option) {
                    return $item->price_point == $allowed_option;
                });

                if ($item_to_push) {
                    $ordered_options_to_return->push($item_to_push);
                }
            });

        return $ordered_options_to_return;
    }

    public function getModel(): string
    {
        return $this->filter_target_model;
    }

    public function title(): string
    {
        return 'Price Point';
    }

    public function modelSlug(Model $model): string
    {
        return '';
    }

    /**
     * Specify attribute on object that represents id
     *
     * @return string
     */
    public function getIdAttribute(bool $for_index = false): string
    {
        return 'price_point';
    }

    /**
     * Specify attribute on object that represents name
     *
     * @return string
     */
    public function getNameAttribute(bool $for_index = false): string
    {
        return 'price_point';
    }

    /**
     * @return array|\Illuminate\Config\Repository|\Illuminate\Contracts\Foundation\Application|\Illuminate\Foundation\Application|mixed
     */
    private function getAllowedOptions()
    {
        // exclude price points that fall below the minimum configured in settings
        $min_value = Settings::get('autotrader-show-price-indicator-min-value');
        $allowed_options = collect(config('automotive.autotrader_price_points'));

        $allowed_options = $allowed_options->map(function ($value) {
            // force labels to upper case first letter of words
            return ucwords(strtolower($value));
        });

        // return the available options which are allowed by the site settings
        return $allowed_options->when(
            $min_value && $allowed_options->search($min_value) !== false,
            fn(Collection $collection) => $collection->splice(0, $allowed_options->search($min_value) + 1)
        );
    }
}
