#!/bin/bash

# Prevent re-running if marker file exists
if [ -f /var/www/html/.setup_done ]; then
    echo "✅ Setup already completed. Skipping..."
    exit 0
fi

echo "🚀 Running initial setup..."

# Ensure necessary directories exist
echo "📂 Creating storage folders..."
mkdir -p storage/framework/cache
mkdir -p storage/framework/cache/data
mkdir -p storage/framework/sessions
mkdir -p storage/framework/views
mkdir -p storage/logs
mkdir -p bootstrap/cache

# Seed uploads folder if it exists
if [ -d "uploads_seeder" ]; then
    echo "📁 Seeding uploads folder..."
    mkdir -p uploads
    cp -r uploads_seeder/* uploads/
else
    echo "⚠️ WARNING: uploads_seeder folder not found. Skipping..."
fi

# Remove old cache files if they exist
echo "🗑️ Clearing Laravel cache..."
rm -f bootstrap/cache/config.php bootstrap/cache/packages.php bootstrap/cache/services.php

# Install Composer dependencies only if vendor folder is missing
if [ ! -d "vendor" ]; then
    echo "📦 Installing Composer dependencies..."
    composer install --prefer-dist --optimize-autoloader
else
    echo "✅ Composer dependencies already installed. Skipping..."
fi

if ! grep -qE "^APP_KEY=[a-zA-Z0-9]" .env; then
    echo "⚠️ APP_KEY is missing or empty. Generating a new one..."
    php artisan key:generate
else
    echo "✅ APP_KEY is set."
fi

# Run migrations with error handling
echo "📜 Running database migrations..."
php artisan migrate --force

# Execute deploy scripts if available
if [ -f shop/deploy.sh ]; then
    echo "🚀 Running shop deployment script..."
    sh shop/deploy.sh
fi

if [ -f cms/deploy.sh ]; then
    echo "🚀 Running CMS deployment script..."
    sh cms/deploy.sh
fi

# Seed database tables only if not seeded before
if [ ! -f "/var/www/html/.seed_done" ]; then
    echo "🌱 Seeding database..."
    php artisan db:seed --force || echo "⚠️ WARNING: Seeding failed, but continuing..."
    php artisan encryption:run || echo "⚠️ WARNING: Encryption failed, but continuing..."
    touch /var/www/html/.seed_done
else
    echo "✅ Database already seeded. Skipping..."
fi

# Install NPM dependencies if missing
if [ ! -d "node_modules" ]; then
    echo "📦 Installing NPM dependencies..."
    npm install
    npm run build
else
    echo "✅ NPM dependencies already installed. Skipping..."
fi

# publish media manager assets
echo "📦 Publishing media manager assets..."
php artisan vendor:publish --tag=media-manager

# Mark setup as complete
touch /var/www/html/.setup_done

echo "✅ Setup completed successfully!"
