<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Tests\TestCase;

class ComponentDiscoveryTest extends TestCase
{
    /**
     * Test component discovery workflow finds Twig files
     */
    public function test_component_discovery_finds_twig_files()
    {
        $response = $this->get('/api/components');

        $response->assertStatus(200);
        $data = $response->json();

        // Verify we have discovered components
        $this->assertIsArray($data['data']);

        // If components exist, verify they have required fields
        if (count($data['data']) > 0) {
            foreach ($data['data'] as $component) {
                $this->assertArrayHasKey('name', $component);
                $this->assertArrayHasKey('filename', $component);
                $this->assertArrayHasKey('path', $component);
                $this->assertStringEndsWith('.twig', $component['filename']);
            }
        }
    }

    /**
     * Test component categorization from directory structure
     */
    public function test_component_categorization()
    {
        $response = $this->get('/api/components');

        $response->assertStatus(200);
        $data = $response->json();

        // Verify categories are created from directory structure
        $this->assertArrayHasKey('categories', $data);
        $this->assertIsArray($data['categories']);

        // If categories exist, verify they have required structure
        if (count($data['categories']) > 0) {
            foreach ($data['categories'] as $category) {
                $this->assertArrayHasKey('name', $category);
                $this->assertArrayHasKey('slug', $category);
                $this->assertArrayHasKey('component_count', $category);
            }
        }
    }

    /**
     * Test component description generation
     */
    public function test_component_description_generation()
    {
        $response = $this->get('/api/components');

        $response->assertStatus(200);
        $data = $response->json();

        // If components exist, verify descriptions are generated
        if (count($data['data']) > 0) {
            foreach ($data['data'] as $component) {
                $this->assertArrayHasKey('description', $component);
                $this->assertNotEmpty($component['description']);
                $this->assertIsString($component['description']);
            }
        }
    }

    /**
     * Test caching of component discovery results
     */
    public function test_component_discovery_caching()
    {
        // First request
        $start1 = microtime(true);
        $response1 = $this->get('/api/components');
        $time1 = microtime(true) - $start1;

        // Second request (should be cached)
        $start2 = microtime(true);
        $response2 = $this->get('/api/components');
        $time2 = microtime(true) - $start2;

        $response1->assertStatus(200);
        $response2->assertStatus(200);

        // Second request should be faster due to caching
        // This is a basic test - actual performance may vary
        $this->assertEquals($response1->json(), $response2->json());
    }
}