---
description: "Update existing component from Figma design"
allowed_tools: ["Read", "Edit", "Write", "Bash", "mcp__figma-desktop__*", "mcp__playwright__*", "TodoWrite", "Glob", "Grep"]
---

# Update Existing Component from Figma

Update an existing Laravel Twig + LESS component with a new Figma design.

## Usage
```
/mtc-update <figma-component-url> <existing-component-name>
```

**Examples:**
```
/mtc-update https://figma.com/design/xxx?node-id=26-215 hero_banner
/mtc-update https://figma.com/design/xxx?node-id=137-156352 cta_boxes
```

## Requirements
- Figma URL with node-id parameter for the new design
- Name of existing component (without path or extension)
- Component must already exist in the codebase

## Component Discovery

If you're unsure which component to update, browse existing components at:
- **Preview URL:** `http://localhost/component`
- **Templates:** `sites/default/templates/includes/`

**Component Locations:**
| Type | Template Directory | CSS Directory |
|------|-------------------|---------------|
| Page Blocks | `templates/includes/page_blocks/` | `css/blocks/` |
| Elements | `templates/includes/elements/` | `css/elements/` |
| Partials | `templates/includes/partials/` | `css/elements/` |
| Page Structure | `templates/includes/page_structure/` | `css/page_structure/` |

## Process

### Step 1: Locate Existing Component

Before making changes:
1. **Find the component files:**
   ```bash
   # Find Twig template
   find sites/default/templates/includes -name "*{component_name}*"

   # Find LESS stylesheet
   find sites/default/css -name "*{component_name}*"
   ```

2. **Read current implementation:**
   - Review the existing Twig template structure
   - Review the existing LESS stylesheet
   - Note any CMS integrations or dummy data patterns
   - Identify variables and props being used

3. **Document what exists:**
   - Current HTML/Twig structure
   - Current CSS classes and BEM naming
   - Current responsive breakpoints
   - Current CMS field mappings

### Step 2: Capture Figma Design

1. **Get Figma design context:**
   ```
   mcp__figma-desktop__get_design_context with nodeId from URL
   ```

2. **Capture reference screenshot:**
   ```
   mcp__figma-desktop__get_screenshot with nodeId
   ```
   Save to: `.frontend/figma-selections/{component_name}-figma-update-{date}.png`

3. **Compare designs:**
   - Identify structural changes (new elements, removed elements)
   - Identify style changes (colors, spacing, typography)
   - Identify responsive behavior changes

### Step 3: Plan Updates

Create a change plan that preserves backward compatibility:

**Preserve:**
- Variable names and CMS data mappings
- Dummy data structure (if exists)
- Any component-specific PHP controller logic
- HTML semantic structure where possible

**Update:**
- Visual styling to match new Figma design
- Layout and spacing
- Responsive behavior
- New design elements

**Document Breaking Changes:**
- If HTML structure must change significantly
- If new required variables are needed
- If CMS field structure needs updating

### Step 4: Update Twig Template

Modify the existing template at:
`sites/default/templates/includes/{type}/{component_name}.twig`

**Guidelines:**
- Keep existing variable names where possible
- Preserve dummy data block structure
- Update HTML structure to match Figma
- Add new variables with sensible defaults
- Update component documentation comments

**Critical Rules (same as extract):**
- NO `asset()` function - use `/sites/default/{{ variable }}`
- Use `.button` class for buttons (no custom button CSS)
- Use `.headingStyle()` mixin for headings
- Use `<div>` for `|raw` content, NEVER `<p>`

### Step 5: Update LESS Stylesheet

Modify the existing stylesheet at:
`sites/default/css/{type}/{component_name}.less`

**Guidelines:**
- Preserve BEM class naming structure where possible
- Map colors to design system variables (variables.less)
- Use `.headingStyle()` mixin for typography
- Update responsive breakpoints to match Figma
- Remove obsolete styles
- Add new styles for new elements

**Design System Integration:**
```less
// Colors - use design system variables
color: var(--primary_color);
background: var(--light_color);

// Typography - use headingStyle mixin
.headingStyle(h2);

// Spacing - use design system variables
padding: var(--top_bottom_gutter) 0;
gap: @gutter_width;

// Responsive - use standard breakpoints
@media @mobile { }
@media @flyout { }
```

### Step 6: Build and Test

1. **Run build:**
   ```bash
   npm run dev
   ```

2. **Check for errors:**
   - Build should complete without errors
   - No LESS syntax errors
   - No missing variable references

### Step 7: Playwright Validation

Test at ALL breakpoints before marking complete:

**Required Breakpoints:**

1. **Mobile (375px)**
   - `mcp__playwright__browser_resize` width: 375, height: 800
   - Navigate to component preview
   - Take screenshot, verify layout

2. **Tablet (768px)**
   - `mcp__playwright__browser_resize` width: 768, height: 1024
   - Verify responsive transition

3. **Desktop (1024px)**
   - `mcp__playwright__browser_resize` width: 1024, height: 768
   - Verify full layout

4. **Wide (1440px)**
   - `mcp__playwright__browser_resize` width: 1440, height: 900
   - Verify max-width constraints

**Test URL:**
- `http://localhost/component/preview/{type}/{component_name}.twig`

**If ANY breakpoint fails:**
- Fix the LESS stylesheet
- Re-run `npm run dev`
- Re-test ALL breakpoints

### Step 8: Document Changes

Add update comment to component files:

**Twig Template:**
```twig
{# Component: {component_name} #}
{# Updated from Figma on {date} #}
{# Changes: Brief description of what changed #}
```

**LESS Stylesheet:**
```less
/* Component: {component_name} */
/* Updated from Figma on {date} */
/* Changes: Brief description of what changed */
```

## Quality Gates

- Visual Accuracy: >= 98% match to Figma
- Console Errors: 0
- Responsive: All 4 breakpoints passing
- Backward Compatible: Existing usages still work
- Design System: 85%+ compliance

## Completion Checklist

Before marking update complete:

- [ ] Existing component files located and backed up (mentally)
- [ ] Figma design captured and analyzed
- [ ] Twig template updated
- [ ] LESS stylesheet updated
- [ ] Build passes without errors
- [ ] Mobile (375px) validated
- [ ] Tablet (768px) validated
- [ ] Desktop (1024px) validated
- [ ] Wide (1440px) validated
- [ ] Console has 0 errors
- [ ] Component documentation updated

## Output Summary

Upon successful update:

**Modified Files:**
- `sites/default/templates/includes/{type}/{component_name}.twig`
- `sites/default/css/{type}/{component_name}.less`

**Created Files:**
- `.frontend/figma-selections/{component_name}-figma-update-{date}.png` (reference)

**Preview:**
- `http://localhost/component/preview/{type}/{component_name}.twig`

## Troubleshooting

**Component not found:**
- Check spelling (use snake_case)
- Try: `find sites/default/templates/includes -name "*{name}*"`
- Browse: `http://localhost/component`

**Build fails:**
- Check LESS syntax errors
- Verify variable references exist in variables.less
- Check for unclosed brackets or missing semicolons

**Visual mismatch:**
- Compare Figma screenshot with Playwright screenshot
- Check responsive breakpoint values
- Verify design system variable values

**Existing functionality broken:**
- Review CMS data mappings
- Check dummy data structure matches
- Verify variable names haven't changed

## Reference

- **Existing Components:** `http://localhost/component`
- **Design System:** `sites/default/css/variables.less`
- **Mixins:** `sites/default/css/base/mixins.less`
- **Extract Command:** `.claude/commands/mtc-extract.md`
