<?php

namespace App\Http\Controllers;

use App\Services\ComponentScannerService;
use App\Services\ComponentRendererService;
use Illuminate\Http\Request;
use Illuminate\Http\Response;

class ComponentPreviewController extends Controller
{
    private ComponentScannerService $scannerService;
    private ComponentRendererService $rendererService;

    public function __construct(
        ComponentScannerService $scannerService,
        ComponentRendererService $rendererService
    ) {
        $this->scannerService = $scannerService;
        $this->rendererService = $rendererService;
    }

    /**
     * Override callAction to skip parent's admin auth requirements.
     */
    public function callAction($method, $parameters)
    {
        return $this->{$method}(...array_values($parameters));
    }

    /**
     * Show component in isolation for preview
     */
    public function show(Request $request, string $path): Response
    {
        try {
            // Find the component
            $component = $this->scannerService->findComponent($path);

            if (!$component) {
                return $this->renderNotFoundError($path);
            }

            // Render component preview
            $html = $this->rendererService->renderComponentPreview($component);

            return response($html, 200, [
                'Content-Type' => 'text/html; charset=UTF-8',
            ]);

        } catch (\Exception $e) {
            return $this->renderServerError($e->getMessage(), $path);
        }
    }

    /**
     * Render 404 error for missing component
     */
    private function renderNotFoundError(string $path): Response
    {
        $html = '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Component Not Found</title>
    <link href="/public/build/app.css" rel="stylesheet">
    <style>
        body {
            font-family: system-ui, -apple-system, sans-serif;
            line-height: 1.5;
            margin: 0;
            padding: 2rem;
            background: #f8f9fa;
        }
        .error-container {
            max-width: 600px;
            margin: 0 auto;
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            padding: 2rem;
            text-align: center;
        }
        .error-title {
            color: #dc3545;
            margin-bottom: 1rem;
        }
        .error-path {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 4px;
            padding: 1rem;
            font-family: monospace;
            margin: 1rem 0;
        }
    </style>
</head>
<body>
    <div class="error-container">
        <h1 class="error-title">Component Not Found</h1>
        <p>The component you requested could not be found.</p>
        <div class="error-path">' . htmlspecialchars($path) . '</div>
        <p><a href="/component">← Back to Component Registry</a></p>
    </div>
    <script src="/public/build/app.js"></script>
</body>
</html>';

        return response($html, 404, [
            'Content-Type' => 'text/html; charset=UTF-8',
        ]);
    }

    /**
     * Render server error
     */
    private function renderServerError(string $errorMessage, string $path): Response
    {
        $html = '<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Component Preview Error</title>
    <link href="/public/build/app.css" rel="stylesheet">
    <style>
        body {
            font-family: system-ui, -apple-system, sans-serif;
            line-height: 1.5;
            margin: 0;
            padding: 2rem;
            background: #f8f9fa;
        }
        .error-container {
            max-width: 800px;
            margin: 0 auto;
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            padding: 2rem;
        }
        .error-title {
            color: #dc3545;
            margin-bottom: 1rem;
        }
        .error-details {
            background: #f8d7da;
            border: 1px solid #f5c6cb;
            border-radius: 4px;
            padding: 1rem;
            margin: 1rem 0;
        }
        .error-path {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 4px;
            padding: 0.5rem;
            font-family: monospace;
            margin: 0.5rem 0;
        }
    </style>
</head>
<body>
    <div class="error-container">
        <h1 class="error-title">Component Preview Error</h1>
        <p><strong>Component:</strong></p>
        <div class="error-path">' . htmlspecialchars($path) . '</div>

        <p><strong>Error:</strong></p>
        <div class="error-details">' . htmlspecialchars($errorMessage) . '</div>

        <p><a href="/component">← Back to Component Registry</a></p>
    </div>
    <script src="/public/build/app.js"></script>
</body>
</html>';

        return response($html, 500, [
            'Content-Type' => 'text/html; charset=UTF-8',
        ]);
    }
}