<?php

namespace MtcPharmacy\Multisite\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Mtc\Shop\Brand as MtcShopBrand;
use Mtc\Shop\Category as MtcShopCategory;
use Mtc\Shop\Item as MtcShopItem;
use Mtc\Cms\Models\Page as CmsPage;
use MtcPharmacy\Multisite\Classes\MultisiteManager;
use MtcPharmacy\Multisite\Classes\MultisiteBaseEntity;
use MtcPharmacy\Multisite\Classes\MultisiteConstants;
use Illuminate\Support\Collection;
use App\MultisiteHelper;


/**
 * Examples:
 *     php artisan multisite:ensure_all_records_are_multisite_ready --entity=cms_page
 */

class EnsureAllRecordsAreMultisiteReadyCommand extends Command
{

    protected $signature = 'multisite:ensure_all_records_are_multisite_ready {--entity=}';

    protected $description = 'Assign relevant entities (those that are neither custom nor base) to the multisite base table.';


    public function __construct()
    {
        parent::__construct();
    }


    public function handle()
    {
        $entity_type = $this->option('entity');

        switch ($entity_type) {
            case MultisiteConstants::ENTITY_TYPE_SHOP_BRAND:
                $this->processShopBrands();
                break;

            case MultisiteConstants::ENTITY_TYPE_SHOP_CATEGORY:
                $this->processShopCategories();
                break;

            case MultisiteConstants::ENTITY_TYPE_SHOP_ITEM:
                $this->processShopItems();
                break;

            default:
                $this->error("Invalid entity type [{$entity_type}].");
                $allowed_entity_types = implode(', ', [
                    MultisiteConstants::ENTITY_TYPE_SHOP_BRAND,
                    MultisiteConstants::ENTITY_TYPE_SHOP_CATEGORY,
                    MultisiteConstants::ENTITY_TYPE_SHOP_ITEM,
                ]);
                $this->warn("Allowed entity types: {$allowed_entity_types}.");
                break;
        }
    }


    private function processShopBrands()
    {
        $all_records = MtcShopBrand::query()
            ->where('deleted', 0)
            ->get()
        ;

        $this->process($all_records, MultisiteConstants::ENTITY_TYPE_SHOP_BRAND);
    }


    private function processShopCategories()
    {
        $all_records = MtcShopCategory::query()
            ->where('deleted', 0)
            ->get()
        ;

        $this->process($all_records, MultisiteConstants::ENTITY_TYPE_SHOP_CATEGORY);
    }


    private function processShopItems()
    {
        $all_records = MtcShopItem::query()
            ->where('deleted', 0)
            ->get()
        ;

        $this->process($all_records, MultisiteConstants::ENTITY_TYPE_SHOP_ITEM);
    }


    private function processCmsPages()
    {
        $all_records = CmsPage::query()->get();

        $this->process($all_records, MultisiteConstants::ENTITY_TYPE_CMS_PAGE);
    }


    private function process(Collection $all_records, string $concrete_entity_type)
    {
        $this->info(__METHOD__);

        foreach ($all_records as $record) {
            $this->info("Processing record id:{$record->id} ...");
            $mm = new MultisiteManager($record);
            $mm->getBaseEntity();
        }

        $this->info('Done.');
    }

}
