---
description: "Generate CMS structure for component and add to example_blocks.json"
allowed_tools: ["Read", "Edit", "Write", "Bash", "TodoWrite"]
---

# CMS Structure Generation (Phase 3)

Generate CMS structure definition for a component and integrate it with the example_blocks.json configuration.

## Usage
```
/mtc-cms-structure <component-name>
```

### Example
```
/mtc-cms-structure results_section
```

## Requirements
- Component Twig template must exist in:
  - `sites/default/templates/includes/page_blocks/`
  - `sites/default/templates/includes/elements/`
  - `sites/default/templates/includes/partials/`
- Component should be added to `example_blocks.twig`

## Process

This command executes the workflow defined in `.frontend/workflows-yaml/cms-structure-generation.yaml`.

**High-Level Steps:**
1. **Analyze Template** - Extract variables and detect field types using AI
2. **Generate Structure** - Create CMS JSON blocks with proper field definitions
3. **Preview & Approve** - Show generated structure for user confirmation
4. **Backup & Insert** - Safely add to example_blocks.json
5. **Update Mappings** - Map CMS data in example_blocks.twig and component template
6. **Validate** - Ensure integration works correctly
7. **Document** - Update component documentation

**Estimated Time:** 3-5 minutes

## 🧠 AI-Powered Features

- **Field Type Inference** - Automatically detects field types from variable names
- **Array Detection** - Identifies repeatable sections from `{% for %}` loops
- **Smart Ordering** - Auto-assigns next available order number
- **Template Updates** - Generates correct CMS data access patterns

## ⚠️ Critical: example_blocks.twig Integration

When updating `example_blocks.twig`, you **MUST**:

1. **REMOVE** `show_dummy_data: true` from the component include
2. **ADD** all CMS field mappings

**Before (from /mtc-extract):**
```twig
{% include 'includes/page_blocks/component.twig' with {
    show_dummy_data: true
}%}
```

**After (from /mtc-cms-structure):**
```twig
{% include 'includes/page_blocks/component.twig' with {
    title: p.pagedata['Component Name'][0]['Title']['value'],
    description: p.pagedata['Component Name'][0]['Description']['value'],
    items: p.pagedata['Component Name Items']
}%}
```

**Dummy data is ONLY for:** `/component/preview/page_blocks/{component}.twig`

## 📝 Template Updates Required

After generating CMS structure, update the component Twig template:

### 1. Textarea Fields (TinyMCE)
Change `<p>` to `<div>` and add `|raw` filter:

```twig
{# Before #}
<p class="component__description">{{ description }}</p>

{# After #}
<div class="component__description">{{ description|raw }}</div>
```

### 2. Array Loops (Dual-Format Support)
Support both dummy data format AND CMS data format:

```twig
{% for item in items %}
    {# Support both dummy data (item.Name.value) and CMS (item['Name']['value']) #}
    {% set item_name = item['Name']['value'] is defined ? item['Name']['value'] : item.Name.value %}
    {% set item_image = item['Image']['value'] is defined ? item['Image']['value'] : item.Image.value %}

    {# Handle CMS image paths #}
    {% if item_image is not empty and item_image starts with 'http' == false and image_folders is defined %}
        {% set item_image = '/' ~ image_folders['cms_images']['original']['path'] ~ '/' ~ item_image %}
    {% endif %}

    <div class="item">
        <img src="{{ item_image }}" alt="{{ item_name }}">
        <h3>{{ item_name }}</h3>
    </div>
{% endfor %}
```

### 3. CMS Field Access Patterns

| Field Type | Pattern |
|------------|---------|
| Single field | `p.pagedata['Block Name'][0]['Field Name']['value']` |
| Array field | `p.pagedata['Block Name Array']` (NO `[0]`) |
| In loop | `item['Field Name']['value']` |

## Output

**Created/Updated Files:**
- `sites/default/templates/structures/example_blocks.json` - CMS structure
- `sites/default/templates/pages/example_blocks.twig` - CMS data mapping
- `sites/default/templates/includes/{dir}/{component}.twig` - Template with |raw filters
- `.frontend/components/{component}.md` - Documentation
- `.backup` file - Safety backup

**Next Steps:**
1. Access CMS admin: `/cms/admin/content_manager/`
2. Add content via CMS
3. View results: `/example-blocks`

## Integration with Workflows

This command can be:
- **Used standalone** - Run independently on any component
- **Part of complete pipeline** - Optional step after `/mtc-extract`

After `/mtc-extract` completes, you can run this command to add CMS management.

## Backup & Safety

- Automatic backup before modifying files
- JSON syntax validation
- Restores from backup if validation fails

## Reference

- **Workflow Definition**: `.frontend/workflows-yaml/cms-structure-generation.yaml` *(source of truth)*
- **CMS Examples**: `sites/default/templates/structures/example_blocks.json`
- **Documentation**: See `.claude/commands/README.md` for detailed information

## Arguments

- `$1` (required) - Component name (e.g., `results_section`)

---

**💡 Tip:** Run this after `/mtc-extract` to add CMS management to your component.
