---
description: "Rebrand design system from full Figma page design"
allowed_tools: ["Read", "Edit", "Write", "mcp__figma-desktop__*"]
---

# Design System Rebrand from Figma

Execute complete design system rebrand from a full Figma page design.

## Usage
```
/mtc-rebrand <figma-url>
```

## Requirements
- Figma design URL (full page or style guide frame with node-id)
- Figma MCP server access
- Write permissions to sites/default/css/ directory

## Tool Usage Guidelines (Token Efficiency)

**⚡ Expected Total Token Usage: 500-2000 tokens (NOT 14k+)**

To prevent excessive token usage during design token extraction:

### ✅ CORRECT Tools for Design Tokens:
1. **`get_screenshot`** - Visual inspection (100-500 tokens per frame)
   - Use for: Typography frames, color palettes, component examples
   - Provides: Clear visual reference for manual extraction

2. **`get_variable_defs`** - Figma variable extraction (200-1000 tokens)
   - Use for: Extracting Figma design variables/tokens
   - Returns: Clean JSON structure with variable definitions
   - Example: `{"typography/primary": "Jost", "color/main": "#003742"}`

3. **Manual Extraction** - Extract directly from screenshots
   - Use when: Figma variables aren't defined in design file
   - Process: Read font names, colors, sizes directly from screenshot

### ❌ WRONG Tool for Design Tokens:
- **`get_code`** - Component code generation (10,000-15,000 tokens!)
  - **NEVER use for design token extraction**
  - Only use for: Phase 1 component extraction (`/mtc-extract`)
  - Why avoid: Generates full React/JSX code with Tailwind classes
  - Result: Massive token waste with irrelevant code generation

### 🚨 Warning Signs:
- **If a single tool call exceeds 5k tokens → You're using the wrong tool**
- **If you see React/JSX code → Stop, you used `get_code` incorrectly**

## Process

This command executes the workflow defined in `.frontend/workflows-yaml/design-system-rebrand.yaml`.

**High-Level Steps:**
1. **Validate Figma URL format**
   - Confirm valid `figma.com/design/` URL structure
   - Extract node IDs from URLs if provided

2. **Extract ALL design tokens** (Token-efficient approach):
   ```
   Step 2.1: Visual Capture (300-500 tokens)
   - Use get_screenshot on typography frame → Identify 2-3 font families
   - Use get_screenshot on color palette frame → Identify 8+ colors
   - Use get_screenshot on component frame → Document form/button styling

   Step 2.2: Variable Extraction (200-1000 tokens)
   - Use get_variable_defs on typography node → Extract font variables
   - Use get_variable_defs on colors node → Extract color variables

   Step 2.3: Manual Extraction (from screenshots)
   - Typography: Extract font names (e.g., "Ivy Mode", "Jost", "Manrope")
   - Colors: Extract hex values (e.g., "#003742", "#E7E7F3")
   - Forms: Extract dimensions (height, padding, border-radius)
   ```
   **⚡ Total: 500-2000 tokens (efficient)**

3. **Update 6 core design system files**
   - variables.less, font.less, typography.less
   - buttons.less, _form_variables.less, form.less

4. **Clean up duplicate font variables**
   - Ensure fonts only in font.less (not variables.less)

5. **Validate build and design system status**
   - Run `npm run dev` to test compilation
   - Verify all status comments added

6. **Mark system as "figma-rebranded"**
   - Add status comment to all 6 files

**Files Updated:**
- `sites/default/css/variables.less` - Color palette, spacing, borders
- `sites/default/css/base/font.less` - Font families (2-3 fonts)
- `sites/default/css/base/typography.less` - Heading styles
- `sites/default/css/base/buttons.less` - Button system
- `sites/default/css/base/_form_variables.less` - Form styling variables
- `sites/default/css/base/form.less` - Form implementation

## Next Steps
- Use `/mtc-extract <component-figma-url>` to build components
- Check `sites/default/css/variables.less` for rebrand status marker

## Token Efficiency Summary

**✅ Correct Execution (500-2000 tokens total):**
- 3x `get_screenshot` calls: ~400 tokens
- 2x `get_variable_defs` calls: ~600 tokens
- Manual extraction from screenshots: ~0 tokens (visual inspection)
- File updates: ~1000 tokens
- **Total: ~2000 tokens ✅**

**❌ Incorrect Execution (14,000+ tokens):**
- 1x `get_code` on typography frame: ~14,400 tokens ❌
- **This is 7x more expensive and provides no additional value**

**Key Rule:** If you're tempted to use `get_code` during rebrand → **STOP** → Use `get_screenshot` instead.

## Reference
- **Workflow Definition**: `.frontend/workflows-yaml/design-system-rebrand.yaml`
- **Token Efficiency**: See "Tool Usage Guidelines" section above
- **Documentation**: See `.claude/commands/README.md` for detailed information