# Design Token Extraction Checklist

**Purpose**: Ensure complete and accurate design token extraction during design system rebrand.

## Overview

This checklist ensures all design tokens are extracted from Figma style guides during the rebrand process. Incomplete extraction leads to missing fonts, colors, or styling that requires rework.

**Critical Principle**: Extract ALL design tokens, not just primary values.

---

## Pre-Extraction Validation

### ✅ Figma Access
- [ ] Figma MCP server connected
- [ ] Valid Figma design URL with `node-id` parameter (recommended)
- [ ] Access to style guide page/frame in Figma
- [ ] Designer has organized style guide with clear sections

### ✅ Figma Style Guide Structure
- [ ] Typography section visible and accessible
- [ ] Colors section visible and accessible
- [ ] Component library with form inputs/buttons visible
- [ ] Spacing/layout standards documented

---

## Typography Extraction (ALL FONTS)

### Step 1: Extract Complete Typography System
**Tool**: Use `get_screenshot` + `get_variable_defs` on typography section node (NOT `get_code` - it's token inefficient)

### Step 2: Identify ALL Font Families
**Expected**: 2-3 distinct font families (NOT just one)

- [ ] **Display Font** identified (for large headings)
  - Examples: IvyMode, Playfair Display, Georgia
  - Usage: Hero headings, large display text
  - Variable: `@display_font`

- [ ] **Primary/UI Font** identified (for interface elements)
  - Examples: Jost, Inter, Roboto
  - Usage: Buttons, navigation, UI elements
  - Variable: `@primary_font`

- [ ] **Body Font** identified (for paragraph text)
  - Examples: Usually same as primary (Jost, Inter)
  - Usage: Paragraph text, body copy
  - Variable: `@body_font`

- [ ] **Secondary/Accent Font** identified (for subheadings)
  - Examples: Manrope, Open Sans, Lato
  - Usage: Subheadings, captions, secondary text
  - Variable: `@secondary_font`

### Step 3: Document Font Details
For each font family:
- [ ] Font name extracted correctly
- [ ] Font weights documented (e.g., 400, 500, 700)
- [ ] Font sizes documented (e.g., 16px, 18px, 24px, 32px)
- [ ] Line heights documented
- [ ] Fallback fonts included in definition

### ✅ Typography Validation
- [ ] **CRITICAL**: Minimum 2 font families identified (typically 3)
- [ ] All font variables will be defined in `font.less`
- [ ] NO font variables will be added to `variables.less`

**⚠️ Common Mistake**: Only extracting 1 font when style guide has 2-3 fonts

---

## Color Palette Extraction (COMPLETE PALETTE)

### Step 1: Extract Complete Color System
**Tool**: Use `get_screenshot` + `get_variable_defs` on colors section node (NOT `get_code` - it's token inefficient)

### Step 2: Identify ALL Color Categories
**Expected**: Minimum 8-12 colors (NOT just 2-4)

#### Primary Brand Colors
- [ ] **Primary Color** (main brand color)
  - Example: `#003742` (dark teal)
  - Variable: `@primary_color`

- [ ] **Secondary Color** (brand accent)
  - Example: `#0887A0` (bright teal)
  - Variable: `@secondary_color`

- [ ] **Primary Color Variants** (2-3 shades)
  - Light, medium, dark variants
  - Variables: `@light_primary`, `@dark_primary`, `@mid_primary`

#### Support/Background Colors (3-4 shades)
- [ ] **Tertiary/Support Color** (light background)
  - Example: `#E7E7F3` (light purple-gray)
  - Variable: `@tertiary_color`
  - Common usage: Input backgrounds, card backgrounds

- [ ] **Support Surface** (surface color)
  - Example: `#EFEFF5`
  - Variable: `@support_surface`

- [ ] **On Support Surface** (lighter surface)
  - Example: `#F5F5FA`
  - Variable: `@on_support_surface`

- [ ] **Light Background Color**
  - Example: `#F8F8F8`
  - Variable: `@light_color`

#### Accent Colors
- [ ] **Third Color** (accent/trust indicator)
  - Example: `#17934C` (Trustpilot green)
  - Variable: `@third_color`

- [ ] **Fourth Color** (secondary accent)
  - Example: `#A1B19E` (secondary green)
  - Variable: `@forth_color`

- [ ] **Secondary Green Variants**
  - Background: `@secondary_green_bg`
  - Surface: `@secondary_green_surface`

#### Text Colors
- [ ] **Primary Text Color**
  - Example: `#232323` (dark gray)
  - Variable: `@text_color`

- [ ] **Light Text Color** (for dark backgrounds)
  - Example: `#FFFFFF`
  - Variable: `@text_color_light`

- [ ] **Dark Title Color** (heading emphasis)
  - Variable: `@dark_title_color`

#### Alert/Status Colors
- [ ] **Success Color** (green)
  - Example: `#458546`
  - Variable: `@success_color`

- [ ] **Error/Alert Color** (red)
  - Example: `#F55D3E`
  - Variable: `@alert_color`

- [ ] **Information Color** (blue)
  - Example: `#2C79AD`
  - Variable: `@information_color`

- [ ] **Warning Color** (yellow/orange)
  - Variable: `@warning_color` (if present)

### ✅ Color Palette Validation
- [ ] **CRITICAL**: Minimum 8 distinct colors extracted (typically 10-12)
- [ ] All color categories represented
- [ ] Hex codes documented accurately
- [ ] Variable names follow existing naming convention

**⚠️ Common Mistake**: Only extracting primary/secondary colors, missing support colors and variants

---

## Component-Specific Styling Extraction

### Form Input Components

#### Step 1: Extract Input Component Styling
**Tool**: Use `get_screenshot` + `get_variable_defs` on input/form component node (NOT `get_code` - it's token inefficient)

#### Step 2: Document Form Styling Details

##### Background & Colors
- [ ] **Input Background Color** extracted
  - Example: `#E7E7F3` (often a light support color)
  - Variable: `@form_background_color`

- [ ] **Input Text Color** extracted
  - Example: `#003742` (primary color)
  - Variable: `@form_text_color`

- [ ] **Placeholder Text Color** extracted
  - Variable: `@form_placeholder_color`

- [ ] **Form Icon Color** extracted
  - Variable: `@form_icons_color`

- [ ] **Focus/Highlight Color** extracted
  - Example: `#0887A0` (secondary color)
  - Variable: `@form_highlight_color`

##### Sizing & Spacing
- [ ] **Input Border Radius** extracted
  - Example: `10px`, `12px`, `16px`
  - Variable: `@form_border_radius`

- [ ] **Input Height/Min Height** extracted
  - Example: `60px`, `65px`, `70px`
  - Variable: `@field_min_height`

- [ ] **Input Padding (Vertical)** extracted
  - Example: `18px`, `20px`, `24px`
  - Variable: `@form_padding_top_bottom`

- [ ] **Input Padding (Horizontal)** extracted
  - Example: `24px`, `26px`, `28px`
  - Variable: `@form_padding_sides`

##### Typography
- [ ] **Input Font Size** extracted
  - Example: `16px`, `18px`
  - Variable: `@form_font_size`

- [ ] **Input Font Weight** extracted
  - Example: `400` (regular)
  - Variable: `@form_font_weight`

- [ ] **Label Font Size** extracted
  - Example: `16px`, `18px`
  - Variable: `@label_font_size`

- [ ] **Label Font Weight** extracted
  - Variable: `@label_font_weight`

##### Borders
- [ ] **Input Border Width** extracted
  - Example: `0px` (borderless), `1px`, `2px`
  - Variable: `@form_border_width`

- [ ] **Input Border Color** extracted
  - Variable: `@form_border_color`

### Button Components

#### Step 1: Extract Button Component Styling
**Tool**: Use `get_screenshot` + `get_variable_defs` on button component node (NOT `get_code` - it's token inefficient)

#### Step 2: Document Button Styling
- [ ] **Button Border Radius** extracted
  - Example: `8px`, `12px`, `16px`
  - Use general `@border_radius` variable

- [ ] **Button Colors** extracted
  - Primary button color
  - Secondary button color
  - Button text color

- [ ] **Button Padding** extracted
  - Vertical and horizontal padding

- [ ] **Button Font Size** extracted

### ✅ Component Styling Validation
- [ ] Form styling completely documented
- [ ] Button styling completely documented
- [ ] All measurements extracted with units (px, rem)
- [ ] All colors mapped to existing color variables

**⚠️ Common Mistake**: Forgetting to extract form input background color and border radius

---

## Spacing & Layout Extraction

### General Spacing
- [ ] **Border Radius (General)** extracted
  - Example: `16px`, `20px`
  - Variable: `@border_radius`
  - Note: Different from form border radius

- [ ] **Grid Spacing** values extracted (if present)

- [ ] **Padding/Margin Standards** extracted (if present)

### ✅ Spacing Validation
- [ ] General border radius documented
- [ ] Spacing scale values documented (if applicable)

---

## Post-Extraction Validation

### Files to Update (6 Total)

#### 1. `sites/default/css/variables.less`
- [ ] Complete color palette added (8+ colors)
- [ ] General border radius updated
- [ ] Spacing values updated
- [ ] Status comment added: `// DESIGN SYSTEM STATUS: figma-rebranded`
- [ ] **CRITICAL**: NO font variables added (they belong in font.less)

#### 2. `sites/default/css/base/font.less`
- [ ] ALL font families defined (2-3 fonts)
- [ ] `@display_font` added
- [ ] `@primary_font` updated
- [ ] `@body_font` updated
- [ ] `@secondary_font` updated
- [ ] Old fonts removed (Barlow, Montserrat, etc.)
- [ ] CSS variables added to `:root`
- [ ] Status comment added

#### 3. `sites/default/css/base/typography.less`
- [ ] Headings use `var(--secondary_font)`
- [ ] Font sizes updated to match Figma
- [ ] Status comment added

#### 4. `sites/default/css/base/buttons.less`
- [ ] Button colors updated from new palette
- [ ] Button border radius updated
- [ ] Uses `var(--primary_font)` for button text
- [ ] Status comment added

#### 5. `sites/default/css/base/_form_variables.less` ⚠️ CRITICAL
- [ ] `@form_background_color` updated
- [ ] `@form_border_radius` updated
- [ ] `@field_min_height` updated
- [ ] `@form_padding_top_bottom` updated
- [ ] `@form_padding_sides` updated
- [ ] `@form_font_size` updated
- [ ] `@label_font_size` updated
- [ ] `@form_text_color` updated
- [ ] `@form_icons_color` updated
- [ ] `@form_border_width` updated
- [ ] `@form_highlight_color` updated
- [ ] CSS variables added to `:root`
- [ ] Status comment added

#### 6. `sites/default/css/base/form.less`
- [ ] Input background uses `var(--form_background_color)`
- [ ] Padding uses `var(--form_padding_sides)`
- [ ] Uses `var(--body_font)` for form text
- [ ] Status comment added

### ✅ Final Validation Checks

#### Design System Status
- [ ] All 6 files contain status comment: `// DESIGN SYSTEM STATUS: figma-rebranded`
- [ ] No duplicate font variables in `variables.less`
- [ ] No old fonts remaining in `font.less`

#### Completeness Checks
- [ ] **Typography**: 2-3 distinct font families defined
- [ ] **Colors**: Minimum 8 colors defined in `variables.less`
- [ ] **Forms**: Background color, border radius, height all updated
- [ ] **Build System**: `npm run dev` compiles successfully

#### Common Mistakes to Avoid
- [ ] ❌ Only extracting 1 font family (need ALL 2-3 fonts)
- [ ] ❌ Only extracting primary color (need complete 8+ color palette)
- [ ] ❌ Forgetting to update `_form_variables.less`
- [ ] ❌ Not updating form border radius to match Figma
- [ ] ❌ Missing input background color update
- [ ] ❌ Adding font variables to `variables.less` instead of `font.less`

---

## Completion Criteria

### ✅ Design System Rebrand Complete When:

1. **Typography System**
   - 2-3 font families defined in `font.less`
   - All fonts have proper fallbacks
   - Display, primary, body, and secondary fonts documented

2. **Color Palette**
   - Minimum 8 distinct colors in `variables.less`
   - All color categories represented (primary, support, accent, text, alert)
   - Color variables follow naming conventions

3. **Component Styling**
   - Form variables completely updated in `_form_variables.less`
   - Button styling updated in `buttons.less`
   - All component-specific styling extracted

4. **System Integration**
   - All 6 files updated with status comments
   - No duplicate font variables
   - Build system compiles successfully
   - Design system status set to "figma-rebranded"

### 🚀 Ready for Component Extraction
Once all criteria met, you can proceed with:
```bash
/mtc-extract <figma-component-url>
```

---

## Quick Reference

### Minimum Requirements
- **Fonts**: 2-3 families
- **Colors**: 8+ distinct colors
- **Files Updated**: 6 files
- **Status Comments**: 6 status comments
- **Build**: Must compile successfully

### Critical Files
1. `variables.less` - Colors, spacing, borders (NO FONTS)
2. `font.less` - ALL font families (2-3 fonts)
3. `_form_variables.less` - Complete form styling

### Common Extraction Pattern
1. Use `get_screenshot` + `get_variable_defs` on typography node → Extract ALL fonts
2. Use `get_screenshot` + `get_variable_defs` on colors node → Extract COMPLETE palette
3. Use `get_screenshot` + `get_variable_defs` on input component → Extract form styling
4. Update 6 files with extracted values
5. Add status comments to all 6 files
6. Validate build system

**⚠️ DO NOT use `get_code`** - It generates 10,000+ tokens of React/JSX code that's not useful for design token extraction.

---

**Version**: 1.0.0
**Last Updated**: 2025-09-30
**Related Workflows**:
- `/mtc-rebrand` command
- `.frontend/workflows-yaml/design-system-rebrand.yaml`