<?php
/**
 * Form Question Class.
 *
 * This code is used to make up Form Questions and their modification
 *
 * @category Forms
 * @package  Mtc_Form_Builder
 * @author   mtc. Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
namespace Mtc\Modules\FormBuilder\Classes;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Schema;

/**
 * Form Question Class. Defines Form Question model.
 * Basic Eloquent model with option relation.
 *
 * @category Forms
 *
 * @author mtc. Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class FormQuestion extends Model
{
    /**
     * @var string Classes associated table
     */
    protected $table = 'forms_questions';

    /**
     * @var bool Disables use of eloquent timestamps
     */
    public $timestamps = false;

    /**
     * @var string Classes fillable params
     */
    protected $fillable = [
        'name',
        'type',
        'fieldset_id',
        'validate',
        'css_class',
        'css_id',
        'placeholder',
        'regex',
        'order'
    ];

    /**
     * FormQuestion::options().
     *
     * Retrieve question options
     */
    public function options()
    {
        return $this->hasMany(FormQuestionOption::class, 'question_id')
            ->orderBy('order', 'ASC');
    }

    public function notifications()
    {
        return $this->hasMany(FormQuestionEmail::class, 'question_id');
    }
    /**
     * FormQuestion::getEmpty().
     *
     * Creates a dummy data array for Vue to process.
     * Eloquent doesn't give out object columns unless it is saved.
     * This function gets object columns and adds relations as empty arrays;
     *
     * @access public
     * @static
     * @return array $empty Empty array with object attributes as keys
     */
    public static function getEmpty()
    {
        $temp = new self();
        $empty = Schema::getColumnListing($temp->getTable());
        $empty = array_fill_keys($empty, '');
        $empty['options'] = [];
        $empty['notifications'] = [];
        return $empty;
    }

    /**
     * Determine which input field type is set based on validation rules provided for current question
     * @return string input field type
     */
    public function getFieldType()
    {
        switch ($this->validate) {
            case "email":
                return 'email';
                break;
            case "numeric":
                return "tel";
                break;
            default:
                return "text";
                break;
        }
    }
}
