<?php

namespace Mtc\Plugins\Clinic\Src;

use Carbon\Carbon;
use Illuminate\Support\Collection;
use Mtc\Modules\Members\Models\MemberNote;

class Flag
{
    /**
     * Gets all patient flags
     *
     * @param $patientID
     * @return Collection
     */
    public static function getPatientFlags($patientID): Collection
    {
        // Load only the columns we need and a trimmed admin_user
        $patientFlags = MemberNote::query()
            ->with(['admin_user:id,name'])
            ->where('member_id', $patientID)
            ->where('is_flag', true)
            ->latest()
            ->get(['id', 'admin_user_id', 'content', 'status', 'flag_due', 'created_at']);

        // Normalize DB flags to the exact fields the UI expects
        $dbFlags = $patientFlags->map(function (MemberNote $note) {
            $author = optional($note->admin_user)->name ?? '';
            $datetime = $note->created_at
                ? Carbon::parse($note->created_at)->format('d.m.Y H:i')
                : null;

            $flagColour = null;
            $flagDueRaw = $note->flag_due; // 'Y-m-d' or null
            $flagDueDisplay = null;

            if (!empty($flagDueRaw)) {
                $today = Carbon::now()->format('Y-m-d');
                if ($flagDueRaw === $today) {
                    $flagColour = 'yellow';
                } elseif ($flagDueRaw < $today) {
                    $flagColour = 'red';
                } else {
                    $flagColour = 'blue';
                }
                $flagDueDisplay = Carbon::parse($flagDueRaw)->format('d/m/Y');
            }

            return [
                'id'         => $note->id,
                'content'    => (string) $note->content,
                'status'     => $note->status,            // 'todo' | 'complete' | null
                'system'     => false,
                'author'     => $author,
                'datetime'   => $datetime,
                'flag_due'   => $flagDueRaw,              // used only for v-if in template
                'flagColour' => $flagColour,              // blue|yellow|red
                'flagDue'    => $flagDueDisplay,          // d/m/Y
            ];
        });

        // Merge with system flags (already normalized)
        return self::getPatientSystemFlags()
            ->merge($dbFlags)
            ->values();
    }

    /**
     * Gets patient system flags
     *
     * @return Collection
     */
    /**
     * Gets patient system flags in the same normalized shape.
     */
    public static function getPatientSystemFlags(): Collection
    {
        return collect([
            [
                'id'         => null,
                'content'    => 'This is an example system flag',
                'status'     => null,
                'system'     => true,
                'author'     => 'System',
                'datetime'   => null,
                'flag_due'   => null,
                'flagColour' => null,
                'flagDue'    => null,
            ],
        ]);
    }
}
