<?php
/**
 * Featured Items
 *
 * Select Featured Items from item table.
 *
 * @author Andrew Morgan <andrew.morgan@mtcmedia.co.uk>
 */

namespace Mtc\Plugins\FeaturedProducts\Classes;

use Illuminate\Database\Eloquent\Model;
use \Item;
use Twig\Environment;

/**
 * Featured Items
 *
 * Select Featured Items from item table.
 *
 * @author Andrew Morgan <andrew.morgan@mtcmedia.co.uk>
 */
class FeaturedProducts extends Model
{
    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'featured_products';

    /**
     * Allow item_id to be set
     *
     * @var Array
     */
    protected $fillable = [
        'item_id'
    ];

    /**
     * FeaturedProducts::get()
     *
     * Get all featured products as set in database
     *
     * @param Object $twig instance of twig
     * @return array
     */
    public static function get(Environment $twig, $return_data = false): array
    {

        // create empty array so return is type array
        $featured_items = [];
        $featured_items_data = [];

        // get all results
        $featured_products = self::getForDisplay();

        // generate HTML
        foreach ($featured_products as $product) {
            $featured_product = new Item();
            $featured_product->Get_Item($product->item_id);

            //Attach category
            $main_category = collect($featured_product->categories)
                ->reject(function ($category) {
                    return $category['name'] == "Doctor";
                })->last();

            $featured_product->main_category = $main_category['name'];


            if($return_data) {

                //Vue
                $item_eloquent = $featured_product->eloquent();
                $featured_product->in_stock = (bool) $item_eloquent->in_stock;
                $featured_product->from_price = (float) $item_eloquent->from_price;
                $featured_product->url = $item_eloquent->url;
                //$featured_product->image = $item_eloquent->getImagePath('small');
                //$featured_product->basket_button_data = $item_eloquent->getProductCardData();
                //$featured_product->main_category = $item_eloquent->getMainCategory();

                $featured_product->price = (int) $featured_product->price;
                $featured_product->sale_price = (int) $featured_product->sale_price;
                
                $featured_items_data[] = (array) $featured_product;
                
                
            }else{

                $featured_item_html = $twig->render('shop/item/small.twig', [
                    'item' => $featured_product
                ]);

                $featured_items[] = $featured_item_html;
            }
            
        }

        if($return_data) {
            return $featured_items_data;
        }else {
            return $featured_items;
        }
    }

     /**
      * Delete featured product when triggered by hook in a result of product delete.
      *
      * @author Vladislavs Ignajtevs <vladislavs.ignatjevs@mtcmedia.co.uk>
      *
      * @param int $id id
      * @return void
      *
      */
    public static function deleteFeatured($id): void
    {
        FeaturedProducts::query()
            ->where('item_id', $id)
            ->delete();
    }

    /**
     *
     * Get Featured Products ordered by "order" and
     * "id descending" for displaying on the page
     *
     * @author Vladislavs Ignatjevs <vladislavs.ignatjevs@mtcmedia.co.uk>
     *
     * @return Array
     */
    public static function getForDisplay()
    {
        return FeaturedProducts::orderBy('order')->orderBy('id', 'desc')->get();
    }
}
