<?php
/**
 * Static Block Class
 *
 * This code is used to make up Static Blocks and their modification
 *
 * @category Plugins
 * @package  Mtc_Static_Blocks
 * @author   mtc. Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
namespace Mtc\Plugins\StaticBlocks\Classes;
use Illuminate\Database\Eloquent\Model;
use Twig\Environment;

/**
 * Static Block Class. Eloquent model class that creates static block
 * functionality. Stores getBlock function which renders block
 *
 * @category Plugins
 * @package  Mtc_Static_Blocks
 * @author   mtc. Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
Class StaticBlocks extends Model
{
    /**
     * @var String Classes associated table
     */
    protected $table = 'static_blocks';

    /**
     * @var boolean Disables use of eloquent timestamps
     */
    public $timestamps = false;

    /**
     * StaticBlocks::getBlock()
     *
     * Retrieve necessary block and render it
     *
     * @access public
     * @static
     * @param  bool             $id   block id
     * @param  Environment $twig Twig render handler
     *
     * @return null
     */
    public static function getBlock($id, Environment $twig)
    {
        $block = StaticBlocks::where('id', $id)
            ->where('published', 1)
            ->first();
        if ($block) {
            self::renderBlock($block, $twig);
        }
    }

    public function content()
    {
        return $this->hasMany(StaticBlocksInstances::class, 'static_block_id');
    }

    /**
     * StaticBlocks::getBlockWithPlaceholders()
     *
     * Retrieve necessary block and replace placeholders with defined values
     *
     * @access public
     * @static
     * @param  bool             $id           block id
     * @param  array            $translations Translation array for placeholders
     * @param  Environment $twig         Twig render handler
     *
     * @return null
     */
    public static function getBlockWithPlaceholders($id, array $translations, Environment $twig)
    {
        $block = StaticBlocks::where('id', $id)->where('published', 1)->first();
        if ($block) {
            if (!empty($translations)) {
                foreach ($translations as $search => $replace) {
                    $block->content = str_replace(
                        '[[[' . $search . ']]]',
                        $replace,
                        $block->content
                    );
                }
            }
            self::renderBlock($block, $twig);
        }
    }

    /**
     * StaticBlocks::renderBlock()
     *
     * Render a static Block
     *
     * @access public
     * @static
     * @param  StaticBlocks     $block Static Block object
     * @param  Environment $twig Twig render handler
     *
     * @return null
     */
    public static function renderBlock(self $block, Environment $twig)
    {
        if ($block && !empty($block->content[0])) {
            if (!empty($block->type)) {
                echo $twig->render(
                    'StaticBlocks/' . $block->type . '.twig',
                    [
                        'content' => $block,
                        'paths' => StaticBlockSizes::getPaths()
                    ]
                );
            }
        }

    }

}
