<?php

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Mtc\Shop\DeliveryMethodRate;
use Mtc\Shop\DeliveryServicesOption;

/**
 * DeliveryMethod.
 *
 * @author Andrius Lekstutis | mtc
 * @copyright 2016 mtc. http://www.mtcmedia.co.uk/
 *
 * @version 2016
 */
class DeliveryMethod extends Model
{
    /**
     * Table name.
     * @var string
     */
    protected $table = "delivery_methods";

    /**
     * Indication about Eloquent timestamps on records.
     * @var bool
     */
    public $timestamps = false;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'zone',
        'name',
        'min',
        'max',
        'cost',
        'service_id',
    ];

    /**
     * define the relationship with table rates for this method
     *
     * @return HasMany
     */
    public function rates(): HasMany
    {
        return $this->hasMany(DeliveryMethodRate::class, 'method_id')
            ->orderBy('weight_min');
    }

    /**
     * relationship with delivery method service
     *
     * @return BelongsTo
     */
    public function service(): BelongsTo
    {
        return $this->belongsTo(DeliveryServicesOption::class, 'service_id');
    }

    /**
     * Define the relationship to items
     * @return BelongsToMany
     */
    public function items(): BelongsToMany
    {
        return $this->belongsToMany(\Mtc\Shop\Item::class, 'items_delivery_methods', 'delivery_method_id');
    }

    /**
     * Get the time until which the countdown timer promises next day delivery along with cutoff time and the next available day
     * returns an Array
     * $arr['timer'] = returns the seconds remaining
     * $arr['delivery_time'] = returns the delivery time string i.e 3:30pm
     * $arr['next_day'] = returns the next day as a string i.e Friday
     * @return array
     */
    public static function getOrderUntilTime($date)
    {
        $return = [
            'timer' => '',
            'delivery_time' => '',
            'next_day' => ''
        ];
        if (empty($date)) {
            return $return;
        }

        $time_date = \Carbon\Carbon::createFromFormat('d/m/Y H:i', $date);

        //Build the hours and minutes that have been set, otherwise fall back to 3:30pm
        if ($time_date) {
            $hours_12  = $time_date->format('g');
            $hours     = $time_date->format('H');
            $minutes   = $time_date->format('i');
            $ampm      = $time_date->format('a');
        } else {
            $hours_12  = 3;
            $hours     = 15;
            $minutes   = 30;
            $ampm      = 'pm';
        }

        //Return just the time string. For use within timer fallback text
        $return['delivery_time'] = $hours . ':' . $minutes;
        $return['delivery_time_12'] = $hours_12 . ':' . $minutes . $ampm;


        $now = \Carbon\Carbon::now();

        $cutoff = \Carbon\Carbon::create(
            $now->format('Y'),
            $now->format('m'),
            $now->format('d'),
            $hours,
            $minutes
        );

        //Set the next available day
        $today = \Carbon\Carbon::now();
        $tomorrow = $today->addDays(1);
        $return['next_day'] =  $tomorrow->format('l');
        $day_of_week = $tomorrow->format('w');

        $now_plus_1 = \Carbon\Carbon::create(
            $now->format('Y'),
            $now->format('m'),
            $now->format('d'),
            $now->format('H') + 1,
            $now->format('i')
        );

        //Get time if the current date is between Monday to Friday
        if ($now->dayOfWeek > 0 && $now->dayOfWeek <= 5 && $now_plus_1 < $cutoff) {
            $return['timer'] = $cutoff->getTimestamp() - $now->getTimestamp();

        } else {
            $tomorrow = $today->addDays(1);
            $return['next_day'] =  $tomorrow->format('l');
            $day_of_week = $tomorrow->format('w');
        }

        if ($day_of_week == '6') {
            $tomorrow = $today->addDays(2);
            $return['next_day'] =  $tomorrow->format('l');
        }

        if ($day_of_week == '0') {
            $tomorrow = $today->addDays(1);
            $return['next_day'] =  $tomorrow->format('l');
        }

        return $return;
    }
}
