// This file is used to test the Vite configuration
// It imports the vite.config.js file and logs the configuration
// This helps verify that the configuration is valid

const fs = require('fs');
const path = require('path');

// Check if vite.config.js exists
console.log('Checking if vite.config.js exists...');
if (fs.existsSync('./vite.config.js')) {
  console.log('vite.config.js exists!');
} else {
  console.error('vite.config.js does not exist!');
  process.exit(1);
}

// Check if package.json has the correct scripts
console.log('\nChecking package.json scripts...');
const packageJson = require('./package.json');
if (packageJson.scripts && packageJson.scripts.dev === 'vite' && packageJson.scripts.build === 'vite build') {
  console.log('package.json scripts are correctly configured!');
} else {
  console.error('package.json scripts are not correctly configured!');
  process.exit(1);
}

// Check if the required dependencies are installed
console.log('\nChecking required dependencies...');
const requiredDeps = ['vite', '@vitejs/plugin-vue2'];
const missingDeps = [];
for (const dep of requiredDeps) {
  if (!packageJson.devDependencies[dep]) {
    missingDeps.push(dep);
  }
}
if (missingDeps.length === 0) {
  console.log('All required dependencies are installed!');
} else {
  console.error(`Missing dependencies: ${missingDeps.join(', ')}`);
  process.exit(1);
}

// Check if the Vite helper function is defined in header.inc.php
console.log('\nChecking for Vite helper function...');
const headerIncPath = './core/includes/header.inc.php';
if (fs.existsSync(headerIncPath)) {
  const headerIncContent = fs.readFileSync(headerIncPath, 'utf8');
  if (headerIncContent.includes("$twig->addFunction(new \\Twig\\TwigFunction('vite'")) {
    console.log('Vite helper function is defined in header.inc.php!');
  } else {
    console.error('Vite helper function is not defined in header.inc.php!');
    process.exit(1);
  }
} else {
  console.error('header.inc.php does not exist!');
  process.exit(1);
}

// Check if main.twig has been updated to use the Vite helper function
console.log('\nChecking if main.twig has been updated...');
const mainTwigPath = './sites/default/templates/layouts/main.twig';
if (fs.existsSync(mainTwigPath)) {
  const mainTwigContent = fs.readFileSync(mainTwigPath, 'utf8');
  if (mainTwigContent.includes('{{ vite(')) {
    console.log('main.twig has been updated to use the Vite helper function!');
  } else {
    console.error('main.twig has not been updated to use the Vite helper function!');
    process.exit(1);
  }
} else {
  console.error('main.twig does not exist!');
  process.exit(1);
}

console.log('\nAll checks passed! The Vite configuration is valid.');
console.log('To complete the migration, run the following commands:');
console.log('1. npm install - to install the new dependencies');
console.log('2. npm run build - to build the assets with Vite');