<?php

namespace Mtc\VehicleValuation;

use App\Http\Requests\ValuationCustomerRequest;
use Carbon\Carbon;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\VehicleValuation\Config\EVAConfig;
use Mtc\VehicleValuation\Contracts\IPRestrictedValuationDriver;
use Mtc\VehicleValuation\Contracts\ValuationDriver;
use Mtc\VehicleValuation\Contracts\ValuationResponse;
use Mtc\VehicleValuation\Exceptions\ValuationRequestFailed;
use Mtc\VehicleValuation\Models\VehicleValuation;

class EVA implements ValuationDriver, IPRestrictedValuationDriver
{
    public function __construct(protected EVAConfig $config)
    {
        //
    }

    public function getValuation(
        string $registration,
        int $mileage,
        ?string $variant = null,
        ?string $cosmetic_condition = null,
        array $data = [],
    ): ValuationResponse {
        $vehicle_data = $this->getVehicleData($registration);

        $response = Http::withHeaders($this->authHeaders())
            ->post($this->endpoint('api/v1/valuation'), [
                'VehicleVRM' => $registration,
                'appraisals' => [
                    [
                        'QuestionSetID' => $this->config->questionSetId(),
                        'CategoryCode' => $this->config->categoryCode(),
                        "BodyStyle" => $vehicle_data['BodyStyle'] ?? '',
                        "VehicleType" => $vehicle_data['VehicleTypeCode'] ?? 'Car',
                        "FuelType" => $vehicle_data['Fuel'] ?? '',
                        'QuestionAnswers' => $this->getAnswersForRequest(
                            ($vehicle_data['VehicleTypeCode'] ?? 'Car') === 'LCV',
                            $mileage
                        ),
                    ],
                ],
            ])
            ->onError(function (Response $response) use ($registration, $mileage) {
                Log::warning('EVA Valuation Request failed', [
                    'message' => $response->body(),
                    'status' => $response->status(),
                    'registration' => $registration,
                    'mileage' => $mileage,
                ]);

                throw new ValuationRequestFailed('Failed to get valuation details from EVA.');
            });

        $valuation_amount = $response->json('Appraisals.0.FinalValuation');

        if (empty($valuation_amount)) {
            throw new ValuationRequestFailed('EVA Valuation request failed');
        }

        return new ValuationResponse(
            provider: 'eva',
            registration: $registration,
            mileage: $mileage,
            vehicleType: 'car',
            retailPrice: null,
            averagePrice: $valuation_amount,
            make: $vehicle_data['Make'],
            model: $vehicle_data['Model'],
            derivative: $vehicle_data['Derivative'],
            fuelType: $vehicle_data['Fuel'],
            engineSize: $vehicle_data['EngineCapacity'],
            bodyType: $vehicle_data['BodyStyle'],
            dateOfValuation: Carbon::now(),
            transmission: $vehicle_data['Transmission'],
            registrationDate: $vehicle_data['DateOfFirstRegistration'],
            colour: $vehicle_data['ExteriorColour'],
            valuation_id: $response->json('ValuationRef'),
        );
    }

    public function getVehicleData(string $registration)
    {
        return Http::withHeaders($this->authHeaders())
            ->get($this->endpoint('api/v1/vehiclelookup/' . $registration))
            ->onError(function (Response $response) use ($registration) {
                Log::warning('EVA Valuation failed to get vehicle details', [
                    'message' => $response->body(),
                    'registration' => $registration,
                ]);
                throw new ValuationRequestFailed('EVA was not able to find vehicle: ' . $response->json('message'));
            })->json();
    }

    public function sendCustomerData(ValuationCustomerRequest $request, VehicleValuation $valuation): bool
    {
        return false;
    }

    protected function endpoint(string $path): string
    {
        return $this->config->isLiveEnvironment()
            ? 'https://www.evaautomotive.co.uk/' . $path
            : 'https://www-test.evaautomotive.co.uk/' . $path;
    }

    protected function accessToken(): string
    {
        return Http::asForm()
            ->withHeaders($this->config->tokenAuthHeaders())
            ->post($this->endpoint('api-auth/oauth2/token'), [
                'grant_type' => 'client_credentials',
            ])
            ->throw()
            ->json('access_token') ?? '';
    }

    protected function authHeaders(): array
    {
        $key = tenant('id') . '-eva-auth-headers';
        return Cache::remember($key, now()->addMinutes(55), fn() => [
            'Authorization' => 'bearer ' . $this->accessToken(),
            'Content-Type' => 'application/json',
        ]);
    }

    private function getAnswersForRequest(bool $isLCV, int $mileage): array
    {
        $data = [
            [
                "QuestionCode" => "lightCondition",
                "Answers" => [
                    ["AnswerCode" => $this->config->conditionValue()]
                ]
            ],
            [
                "QuestionCode" => "vehicleUse",
                "Answers" => [
                    ["AnswerCode" => "positive"]
                ]
            ],
            [
                "QuestionCode" => "privateHire",
                "Answers" => [
                    ["AnswerCode" => "positive"]
                ]
            ],
            [
                "QuestionCode" => "emergencyVehicle",
                "Answers" => [
                    ["AnswerCode" => "positive"]
                ]
            ],
            [
                "QuestionCode" => "mot",
                "Answers" => [
                    ["AnswerCode" => "3"]
                ]
            ],
            [
                "QuestionCode" => "serviceHistory",
                "Answers" => [
                    ["AnswerCode" => "positive"]
                ]
            ],
            [
                "QuestionCode" => "V5Logbook",
                "Answers" => [
                    ["AnswerCode" => "positive"]
                ]
            ],
            [
                "QuestionCode" => "keys",
                "Answers" => [
                    ["AnswerCode" => "positive"]
                ]
            ],
            [
                "QuestionCode" => "mechanicalIssues",
                "Answers" => [
                    ["AnswerCode" => "positive"]
                ]
            ],
            [
                "QuestionCode" => "warranty",
                "Answers" => [
                    ["AnswerCode" => "positive"]
                ]
            ],
            [
                "QuestionCode" => "totalLoss",
                "Answers" => [
                    ["AnswerCode" => "positive"]
                ]
            ],
            [
                "QuestionCode" => "isImport",
                "Answers" => [
                    ["AnswerCode" => "positive"]
                ]
            ],
            [
                "QuestionCode" => "leftHandDrive",
                "Answers" => [
                    ["AnswerCode" => "positive"]
                ]
            ],
            [
                "QuestionCode" => "keepOwnReg",
                "Answers" => [
                    ["AnswerCode" => "positive"]
                ]
            ],
            [
                "QuestionCode" => "vehicleMileage",
                "Answers" => [
                    [
                        "AnswerValue" => $mileage,
                        "AnswerCode" => "value"
                    ]
                ]
            ]
        ];

        if ($isLCV) {
            $data[] = [
                "QuestionCode" => "vehicleUsage",
                "Answers" => [
                    [
                        "AnswerCode" => $this->config->lcvUsageValue(),
                    ],
                ],
            ];
        }

        return $data;
    }
}
