<?php

namespace Mtc\VehicleValuation;

use App\Http\Requests\ValuationCustomerRequest;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\VehicleValuation\Config\HPIConfig;
use Mtc\VehicleValuation\Contracts\ValuationDriver;
use Mtc\VehicleValuation\Contracts\ValuationResponse;
use Mtc\VehicleValuation\Exceptions\ValuationRequestFailed;
use Mtc\VehicleValuation\Models\VehicleValuation;

class HPI implements ValuationDriver
{
    public function __construct(protected HPIConfig $config)
    {
        //
    }

    public function getValuation(
        string $registration,
        int $mileage,
        ?string $variant = null,
        ?string $cosmetic_condition = null,
        array $data = [],
    ): ValuationResponse {
        $xml = $this->getVehicleData($registration, $mileage);
        if (!$xml instanceof \SimpleXMLElement) {
            throw new ValuationRequestFailed('HPI response was empty or invalid.');
        }

        $payload = json_decode(json_encode($xml), true) ?? [];

        //ConditionBasedValuation
        $cbv = Arr::wrap(data_get(
            $payload,
            'Asset.SupplementaryAssetData.CAPValuation.MileageAndConditionBasedValuation.ConditionBasedValuations'
            . '.ConditionBasedValuation',
            []
        ));
        $averageValuation = collect($cbv)->firstWhere('Description', 'Average');
        $cleanValuation = collect($cbv)->firstWhere('Description', 'Clean');
        $belowAverageValuation = collect($cbv)->firstWhere('Description', 'Below Average');

        $valuation_amount = data_get($averageValuation, 'TradeValue');

        if (empty($valuation_amount)) {
            throw new ValuationRequestFailed('HPI Valuation request failed');
        }

        return new ValuationResponse(
            provider: 'hpi',
            registration: $registration,
            mileage: $mileage,
            vehicleType: 'car',
            retailPrice: (float)data_get(
                $payload,
                'Asset.SupplementaryAssetData.CAPValuation.MileageAndConditionBasedValuation.RetailPrice'
            ),
            averagePrice: (float)$valuation_amount,
            cleanPrice: (float)data_get($cleanValuation, 'TradeValue'),
            belowPrice: (float)data_get($belowAverageValuation, 'TradeValue'),
            make: data_get($payload, 'Asset.PrimaryAssetData.DVLA.Make.Description')
            ?? data_get($payload, 'Asset.PrimaryAssetData.SMMT.Make.Description'),
            model: data_get($payload, 'Asset.PrimaryAssetData.DVLA.Model.Description')
            ?? data_get($payload, 'Asset.PrimaryAssetData.SMMT.Model.Description'),
            derivative: $variant
            ?? data_get($payload, 'Asset.SupplementaryAssetData.CAPValuation.CAPVariant')
            ?? data_get($payload, 'Asset.PrimaryAssetData.SMMT.Model.Description'),
            fuelType: data_get($payload, 'Asset.PrimaryAssetData.DVLA.Engine.Fuel.Description'),
            engineSize: data_get($payload, 'Asset.PrimaryAssetData.DVLA.Engine.Size') ?: null,
            bodyType: data_get($payload, 'Asset.PrimaryAssetData.DVLA.Body.Description'),
            dateOfValuation: Carbon::now(),
            transmission: data_get($payload, 'Asset.PrimaryAssetData.SMMT.Transmission.Description'),
            registrationDate: data_get($payload, 'Asset.PrimaryAssetData.DVLA.KeyDates.FirstRegistered.Date'),
            colour: data_get($payload, 'Asset.PrimaryAssetData.DVLA.Body.Colour.Current.Description'),
            valuation_id: (string)data_get($payload, 'Asset.SupplementaryAssetData.CAPValuation.CAPId')
        );
    }

    protected function getVehicleData($registration_number, $mileage): ?\SimpleXMLElement
    {
        try {
            $response = $this->call($this->getVehicleRequest($registration_number, $mileage))->body();
            $xml = simplexml_load_string($response);

            return $xml
                ->children('http://schemas.xmlsoap.org/soap/envelope/')
                ->Body
                ->children('http://webservices.hpi.co.uk/SupplementaryEnquiryV1')
                ->EnquiryResponse
                ->RequestResults;
        } catch (Exception $exception) {
            Log::error(__CLASS__ . '::' . __FUNCTION__ . '() ' . $exception->getMessage());
            return null;
        }
    }

    public function sendCustomerData(ValuationCustomerRequest $request, VehicleValuation $valuation): bool
    {
        return false;
    }

    protected function call(array $request_data)
    {
        $response = Http::withHeaders($this->getHeaders())
            ->timeout(20)
            ->withBody($this->getEnvelope($request_data), 'text/xml')
            ->post($this->config->endpoint());

        if ($response->failed()) {
            Log::error('HPI valuation call failed', [
                'endpoint' => $this->config->endpoint(),
                'request_data' => $request_data,
                'status_code' => $response->status(),
                'response' => $response->body(),
            ]);
            throw new ValuationRequestFailed('Failed to get valuation details from HPI.');
        }

        return $response;
    }

    protected function getVehicleRequest($registration_number, $mileage): array
    {
        return [
            'subscriber_details' => [
                'customer_code' => $this->config->customerCode(),
                'initials' => $this->config->initials(),
                'password' => $this->config->password(),
            ],
            'asset' => [
                'vrm' => $registration_number,
                'reference' => '',
                'mileage' => $mileage ? (int)ceil($mileage / 1000) : null,
            ],
            'primary_product' => $this->config->primaryProductCode(),
            'supplementary_product' => $this->config->supplementaryProductCode(),
        ];
    }

    protected function getHeaders(): array
    {
        return [
            'Content-Type' => 'text/xml; charset=utf-8',
            'SOAPAction' => 'http://webservices.hpi.co.uk/SupplementaryEnquiryV1/enquire',
        ];
    }

    protected function getEnvelope(array $request_data): string
    {
        return view('valuation::hpi_enquiry_valuation_request', ['data' => $request_data])->render();
    }
}
