<?php

namespace Mtc\VehicleValuation;

use App\Http\Requests\ValuationCustomerRequest;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\VehicleValuation\Config\GoodbyeCarConfig;
use Mtc\VehicleValuation\Contracts\ValuationResponse;
use Mtc\VehicleValuation\Exceptions\ValuationRequestFailed;
use Mtc\VehicleValuation\Models\VehicleValuation;

class GoodbyeCar implements Contracts\ValuationDriver
{
    public function __construct(protected GoodbyeCarConfig $config)
    {
        //
    }

    public function getValuation(
        string $registration,
        int $mileage,
        ?string $variant = null,
        ?string $cosmetic_condition = null,
    ): ValuationResponse {
        $valuation_response = Http::post($this->endpoint('api/request-valuation'), [
                'vrm' => $registration,
                'mileage' => $mileage,
            ])
            ->onError(function (Response $response) use ($registration, $mileage) {
                Log::warning('Goodbye Car Valuation Request failed', [
                    'message' => $response->body(),
                    'status' => $response->status(),
                    'registration' => $registration,
                    'mileage' => $mileage,
                ]);
                throw new ValuationRequestFailed('Failed to get valuation details from Goodbye Car.');
            });

        return new ValuationResponse(
            provider: 'goodbye-car',
            registration: $registration,
            mileage: $mileage,
            make: $valuation_response['vehicle']['manufacturer'] ?? '',
            model: $valuation_response['vehicle']['model'] ?? '',
            bodyType: $valuation_response['vehicle']['body'] ?? '',
            colour: $valuation_response['vehicle']['colour'] ?? '',
            engineSize: $valuation_response['vehicle']['engine_size'] ?? '',
            fuelType: $valuation_response['vehicle']['fuel'] ?? '',
            transmission: $valuation_response['vehicle']['transmission'] ?? '',
            averagePrice: $valuation_response['valuation'] ?? 0,
            cleanPrice: $valuation_response['valuation_high'] ?? 0,
            belowPrice: $valuation_response['valuation_low'] ?? 0,
            valuation_id: $valuation_response['id'] ?? '',
            vehicleType: 'car',
            retailPrice: null,
        );
    }

    public function sendCustomerData(ValuationCustomerRequest $request, VehicleValuation $valuation): bool
    {
        try {
            $response = Http::post($this->endpoint('api/customer-info'), [
                'id' => $valuation->toArray()['valuation_id'],
                'name' => $request['first_name'] . ' ' . $request['last_name'],
                'telephone' => $request['phone'],
                'email' => $request['email'],
                'postcode' => $request['postcode'],
            ])
                ->onError(function (Response $response) {
                    Log::warning('Goodbye Car Customer Info Request failed', [
                        'message' => $response->body(),
                        'status' => $response->status(),
                    ]);
                    throw new ValuationRequestFailed('Failed to send customer info to Goodbye Car.');
                });

            if (
                !empty($response)
                && array_key_exists('status', $response->json())
                && $response->json('status') == 'success'
            ) {
                return true;
            }
        } catch (\Exception $exception) {
            Log::error('Exception: ' . $exception->getMessage());
        }

        return false;
    }

    protected function endpoint(string $path): string
    {
        $url = app()->environment('production')
            ? 'https://www.goodbyecar.uk/' . $path
            : 'http://www.goodbyecar.mtcdevserver4.com/' . $path;

        return $url . '?' . http_build_query([
            'api_token' => $this->config->apiKey()
        ]);
    }
}
